<?php
/**
* @package direct-as-a-service
* @subpackage libraries
*/

load_helpers('array', 'date', 'inflector');


/**
* This is a functional accessor for {@link Error_helper::should_be_an_x}, to be used when there is not an error helper object in scope.  
* Using the error helper object directly is the preferred best practice when possible.
* @uses Error_helper::should_be_an_x
* @param string Description of what sort of value was expected
* @param mixed The invalid value
* @return false
*/
if(!function_exists('should_be')){
	function should_be($what_it_should_be, $value){
		$error_helper = new Error_helper();
		return $error_helper->should_be_an_x($value, $what_it_should_be, 1);
	}
}

if(!function_exists('get_error_helper')){
	$ERROR_HELPER;
	function get_error_helper(){
		global $ERROR_HELPER;
		if(!is_a($ERROR_HELPER, 'Error_helper'))
			$ERROR_HELPER = new Error_helper();
		return $ERROR_HELPER;
	}
}

/**
* Library to help developers write more informative, concise, and consistent error messages.
*
* Descriptive error messages are one of the most useful tools you can add to your code to assist in debugging and aid maintainability, but
* generating descript error messages on a case-by-case basis can be time-consuming and tedious.  This class endeavours to make it easier to write
* the most common error messages as readably as possible, and in as few lines of code as possible.
*
* <strong>Usage example</strong>
* To demonstrate the use of this class, we'll take a look at one of the most commonly used methods, {@link should_be_an_x()}.  {@link should_be_an_x()}
* is used to generate an error message that expresses that a value was expected to match certain criteria, and describes what the value was instead.
* It makes use of the {@link describe()} method to express what was found; this is a useful method on its own, which attempts to describe a value as specifically as possible
* in words.  {@link describe} differentiates between boolean false, null, and the empty string, distinguishing between numberic strings and values that are currently cast as
* an int/float, specifies the class if the value is an object, etc.
*
* Like any of our methods that end with an "_x", the {@link should_be_an_x()} method is set up to work with {@link __call} so that you can replace the "_x" portion
* of the method with a description of what you were expecting.
* <code>
* $invalid_value = new Date();
* //calling on the method directly - slightly more efficient, but less readable.
* $error_helper->should_be_an_x('array of strings', $invalid_value);
* //preferred method -- more readable, and will result in the same message.
* $error_helper->should_be_an_array_of_strings($invalid_value);
*
* //both methods trigger a warning saying "I expected an array of strings, but you gave me an instance of the Date class";
* </code>
* 
* Errors generated by {@link should_be_an_x} default to the level specified in {@link error_levels}, which by default is E_USER_WARNING.  You could extend this class
* and specify a different value in order to change this on a global level for your application.  Alternately, you could make use of the {@link notice}, {@link warning},
* or {@link fatal_error} variables to override the error level on a case-by-case basis.
* <code>
* //triggers an E_USER_ERROR even though should_be errors are usually E_USER_WARNING
* $error_helper->fatal_error->should_be_an_array_of_string($invalid_value);
* </code>
*
* <strong>Offsets -- or, correctly identifying file and line numbers</strong>
* PHP errors cite the file and line number where an error took place, and for developer-triggered errors, that file and line number is always where {@link trigger_error}
* was called.  Without intervention, that means that any errors triggered by the {@link Error_helper} class will always cite the {@link _trigger_error} method as the
* source of the error.  In order to prevent this, the class tracks the number of method calls that take place once the Error_helper is called, and stores that number
* in a global called {@link $EXCEPTION_OFFSET} so that the error handler can step back through the backtrace to find the point at which the the Error_helper was called.
* If you're working in a Codeigniter context and using the {@link ICARUS_Exceptions} extension, this is already taken care of for you; if you need to add this to your 
* own error handler, {@link ICARUS_Exceptions::show_php_error()} will serve as a good example.  See {@link _trigger_error} for more information.
*
* @author M. Gibbs <gibbs_margaret@bah.com>
* @package direct-as-a-service
* @subpackage libraries
*/
class Error_helper{

	/**
	* An instance of the error helper where all error levels are set to E_USER_NOTICE
	*
	* See {@link fatal_error} for more information on how to use this.
	*
	* @var Error_helper
	*/	
	var $notice;
	
	/**
	* An instance of the error helper where all error levels are set to E_USER_WARNING.
	*
	* See {@link fatal_error} for more information on how to use this.
	*
	* @var Error_helper
	*/	
	var $warning;
	
	/**
	* An instance of the error helper where all error levels are set to E_USER_ERROR.
	*
	* Having this available as a public variable allows you to gracefully temporarily switch the normal error level of one of the error methods.
	* For example, if it was so vital that a parameter for a function be of a certain type that you wanted to halt the script if it was wrong, you could write: 
	*
	* <code>
	* $this->error_helper->fatal_error->should_be_a_string($critical_var);
	* </code>
	*
	* This would override the normal error level for the should_be_an_x method and trigger a fatal error.  Note that this should be used sparingly
	* and only as needed; if you routinely started writing $this->error_helper->warning->should_be_a_string($normal_var) instead of the standard
	* $this->error_helper->should_be_a_string($normal_var), you would lose the ability to easily change the level of all should_be errors by 
	* changing the error level in the {@link error_levels} array.
	*	
	* @var Error_helper
	*/
	var $fatal_error;
	
	/**
	* Maps the error types in this class to their PHP error level.
	* Mapping them here means that if we suddenly decide that all 'should_be_an_x' errors should be fatal, we just need to change value in this array.
	* You could also extend this class in your application to change error levels in just your application.
	* @var array
	*/
	var $error_levels = array(	'should_be_an_x' => E_USER_WARNING,
								'no_such_property' => E_USER_WARNING,
								'no_such_method' => E_USER_WARNING,
								'notice' => E_USER_NOTICE,
								'warning' => E_USER_WARNING,
								'error' => E_USER_ERROR
								);


	/** 
	* Set up default class variables.
	*/
	function __construct($error_level = null){
		if(empty($error_level)){
			$this->notice = new Error_helper(E_USER_NOTICE);
			$this->warning = new Error_helper(E_USER_WARNING);
			$this->fatal_error = new Error_helper(E_USER_ERROR);
		}else{
			foreach($this->error_levels as $description => $existing_error_level){
				$this->error_levels[$description] = $error_level;
			}
		}
	}														
														
	
	/**
	* Basic wrapper to trigger a notice.
	* This is functionally the same as calling PHP's trigger_error($your_message, E_USER_NOTICE).
	*	
	* @uses _trigger_error
	*
	* @param string $message
	* @param int $offset (optional)
	* @return false
	*/
	function notice($message, $offset=0){
		return $this->_trigger_error($message, 'notice', $offset+1);
	}
	
	/**
	* Basic wrapper to trigger a warning.
	* This is the same as calling PHP's trigger_error($your_message, E_USER_WARNING).
	* 
	* @uses _trigger_error
	*
	* @param string $message
	* @param int $offset (optional)
	* @return false
	*/
	function warning($message, $offset=0){
		return $this->_trigger_error($message, 'warning', $offset+1);
	}
	
	/**
	* Basic wrapper to trigger an error.
	* This is the same as calling PHP's trigger_error($your_message, E_USER_ERROR).
	* 
	* @uses _trigger_error
	*
	* @param string $message
	* @param int $offset (optional)
	* @return false
	*/
	function error($message, $offset=0){
		return $this->_trigger_error($message, 'error', $offset+1);	
	}
	
	/**
	* Basic wrapper to trigger a fatal error.
	* Since the CI error handler doesn't allow for a user-generated fatal error, 
	* this methods exits out after calling on {@link error()}
	* 
	* @uses _trigger_error
	*
	* @param string $message
	* @param int $offset (optional)
	* @return false
	*/
	function fatal($message, $offset=0){
		$this->error($message, $offset+1);		
		exit;
	}
	
	/**
	* Generates an informative error message for invalid input.
	*
	*	Reporting invalid input is one of the most common error tasks we've come across.  Helpful error messages for invalid input describe both
	* what was expected and what was received in its place.  To cut down on the amount of typing (and standardize the error message), this function allows
	* you to simply pass the value that you received and a string describing what you expected.  Thanks to some magical method handling in {@link __call}, you can
	* make this even more readable by including the description of what you expected in the method call instead of the parameter.
	*
	* <code> 
	* $invalid_value = "I am an invalid value";
	*	$this->error->should_be_an_x($invalid_value, 'array');
	*	$this->error->should_be_an_array($invalid_value); //equivalent, preferred for readability
	* </code>
	*	
	* The code above will output an error message that says 'You gave me "I am an invalid value" when I expected an array'.  This method makes use of the 
	* {@link describe} method to attempt to describe the $what_it_was parameter as helpfully as possible (for example, distinguishing between NULL and FALSE).
	*
	* If you're using an error handler that understands the {@link $EXCEPTION_OFFSET} global, the error message will correctly cite the line number 
	* and file where you called the should_be method correctly.  If you're not using a custom error handler, it will always reference the {@link trigger_error}
	* call in {@link _trigger_error}. You may want to consider writing a custom error handler that can handle this; at the time of this writing, the error 
	* handler in /wwwroot/htdocs/webapps/codeigniter/2.0/Common.php is a good example.
	*
	* @uses _trigger_error
	* @uses describe
	*
	* @param mixed $what_it_was The actual invalid value (the array, object, etc).  This method will come up with a way to describe it, so you don't need to.
	* @param string $should_be A description of the kind of value you were expecting to get.
	* @param int $offset (optional)
	* @return false
	*/
	function should_be_an_x($what_it_was, $should_be, $offset=0){
		$offset = 1 + $offset;
		$message = 'You gave me '.$this->describe($what_it_was).' when I expected '.indefinite_article($should_be).' '.$should_be;
		return $this->_trigger_error($message, 'should_be_an_x', $offset);																	 
	}
	
	/**
	* Triggers an error expressing that a value fell outside of a valid numeric range.
	*
	* This is a wrapper for {@link should_be_an_x} and will generate an error message in the same manner and at the same level as that method.
	*
	* <code>
	* $invalid_input = 3.14;
	* $this->error_helper->should_be_within_range($invalid_input, 5, 10);
	* </code>
	*
	* The example above will output an error that says "I expected a number between 5 and 10, but you gave me 3.14".
	*
	* @param mixed $what_it_was
	* @param int|float|double|long $range_min
	* @param int|float|double|long $range_max
	* @param int $offset (optional)
	* @return false
	*/
	function should_be_within_range($what_it_was, $range_min, $range_max, $offset=0){
		$message = 'number between '.$this->describe($range_min).' and '.$this->describe($range_max);
		return $this->should_be_an_x($what_it_was, $message, $offset+1);
	}
		
	/**
	* Triggers an error expressing that the value was invalid for a class variable.
	*
	* This is a almost exactly like {@should_be_an_x}, but you can use it to be a little more informative when the invalid input involved is part of
	* a class getter or setter. Note that it uses magical __call functionality just like {@link should_be_an_x}.
	*
	* <code>
	* $slytherin = new HogwartsHouse();
	* $not_a_hogwarts_mascot = array('a', 'b', 'c');
	* $slytherin->mascot = $not_a_hogwarts_mascot;
	*
	* //if the setter for HogwartsHouse had a type check, it could use either of these statements to express an invalid type error:
	* $this->error->property_value_should_be_an_x('mascot', $slytherin, $not_a_hogwarts_mascot, 'name of an animal');
	* $this->error->property_value_should_be_a_name_of_an_animal('mascot', $slytherin, $not_a_hogwarts_mascot);
	* </code>
	*
	* The statements above would output an error message saying "The value of HogwartsHouse::mascot should be a name of an animal,
	* but you gave me an array".
	*
	* @uses _trigger_error
	*
	* @param string $property
	* @param string $class
	* @param mixed $what_it_was
	* @param string $should_be
	* @param int $offset (optional)
	* @return false
	*/
	function property_value_should_be_an_x($property, $class_or_object, $what_it_was, $should_be, $offset=0){	
		$offset = 1 + $offset;
		if(is_object($class_or_object)) $class = get_class($class_or_object);	else $class = $class_or_object;
		$message = 'The value of '.$class.'::'.$property.' should be '.indefinite_article($should_be).' '.$should_be.',';
		$message .= ' but you gave me '.$this->describe($what_it_was);
		return $this->_trigger_error($message, 'should_be_an_x', $offset);		
	}

	/**
	* Triggers an error expressing that a value is not a valid name for the column of a table.
	*
	* This is a wrapper for {@link should_be_an_x} and will generate an error message in the same manner and at the same level as that method.
	*
	* <code>
	* $invalid_input = 'I am not a column name';
	* $this->error_helper->should_be_a_column_for_table($invalid_input, 'BAHStaff.employee');
	* </code>
	*
	* The example above will output an error that says "I expected a column for the BAHStaff.employee table, but you gave me 'I am not a column name'".
	*
	* @param mixed $what_it_was
	* @param string $table_name
	* @param int $offset (optional)
	* @return false
	*/	
	function should_be_a_column_for_table($what_it_was, $table_name, $offset=0){
		$message = 'valid column for the '.$table_name.' table';
		return $this->should_be_an_x($what_it_was, $message, $offset+1);
	}
	
	function should_be_an_entity_for_this_model($what_it_was, &$model_or_model_name, $offset=0){
		if(is_string($model_or_model_name))
			$model_name = $model_or_model_name;
		elseif(is_a($model_or_model_name, 'Active_record_model'))
			$model_name = $model_or_model_name->model_alias();
		
		$message = 'entity';
		if(!empty($model_name))
			$message = $model_name.' entity';

		return $this->should_be_an_x($what_it_was, $message, $offset+1);
	}	
	
	function value_exceeds_max_length_for_table_column($value, $max_length, $column, $table, $offset=0){
		$message = 'Table column '.$table.'.'.$column.' has a max length of '.$max_length.', but you gave me a value of length '.mb_strlen($value).'.  ';
		$message .= 'The contents of this column may get cut off by the database.';
		return $this->notice($message, $offset+1);
	}
	
	function property_does_not_exist($property, $class, $offset=0){
		if(is_object($class)) $class = get_class($class);
		if(!is_string($class)) $class = $this->describe($class);
		if(!is_string($property)) $property = $this->describe($property);
		
		$message = 'There is no property named "'.$property.'" for class '.$class;
		return $this->_trigger_error($message, 'no_such_property', $offset+1);
	}
	
	function method_does_not_exist($method, $class, $offset=0){
		$message = 'There is no method named "'.$method.'" for class '.$class;
		return $this->_trigger_error($message, 'no_such_method', $offset+1);
	}
	
	function relationship_does_not_exist($relationship_name, $model_name, $offset=0){
		$message = 'The '.$model_name.' model does not have a relationship called '.$this->describe_as_text($relationship_name);
		return $this->warning($message, $offset+1);
	}

	/**
	* Determines the PHP error level that should be used for a given error method.
	* 
	* @see http://us3.php.net/manual/en/errorfunc.constants.php
	*
	* @param string Descriptor of the error method as specified in {@link error_levels}, or the error method name itself.
	* @return int PHP error level
	*/
	function level($error_description){
		if(array_key_exists($error_description, $this->error_levels))
			return $this->error_levels[$error_description];
		if($this->is_method_name_for_should_be_an_x($error_description))
			return $this->error_levels['should_be_an_x'];
		
		return E_USER_ERROR;	//default to something massive that hopefully people will notice;
	}
	
	/**
	* Describes a value in a way that will be useful in an error message.
	* For example, nulls will be described as the word 'null' instead of an empty string, booleans will have be described as 'boolean true' or 
	* 'boolean false' instead of as 0 or 1, strings will always appear in quotes, etc.
	* @param mixed value
	* @return string Description
	*/
	public function describe($value){
		if(is_null($value))
			$value = 'NULL';
		elseif(is_bool($value))
			$value = $value ? 'boolean TRUE' : 'boolean FALSE';
		elseif(is_array($value)){
			if(empty($value)) 
				$value = 'an empty array';
			elseif(function_exists('validates_as')){
				//if we don't have any objects/complicated things and this is small, describe the actual array values
				$first_element = first_element($value);
				$is_displayable = (count($value) < 50 && (is_scalar($first_element) || is_bool($first_element) || is_null($first_element)));
				foreach($value as $displayable_value){
					if(!$is_displayable) break;
					$is_displayable = $is_displayable && (!is_object($displayable_value) && !is_array($displayable_value) && !is_resource($displayable_value));
				}
				if($is_displayable){
					$value = strip_from_beginning('[sprp]', strip_from_end('[/sprp]', trim(sprp_for_log($value))));
				}
				elseif(is_bool($first_element) && validates_as('array_of_booleans', $value)) $value = 'an array of booleans';
				elseif(is_int($first_element) && validates_as('array_of_integers', $value)) $value = 'an array of ints';
				elseif(is_numeric($first_element) && validates_as('array_of_numerics', $value)) $value = 'an array of numbers';
				elseif(is_array($first_element) && validates_as('array_of_arrays', $value)) $value = 'an array of arrays';
				elseif(is_string($first_element) && validates_as('array_of_strings', $value)) $value = 'an array of strings';
				elseif(is_object($first_element) && validates_as('array_of_objects', $value)) $value = 'an array of objects';
				else $value = 'an array';
			}
			else
				$value = 'an array';
		}
		elseif(is_a($value, 'Entity')){
			$id = $value->id();
			$type = get_class($value);
			if(!empty($id))
				$value = indefinite_article($type).' '.$type.' entity with an id of '.$value->id();
			else
				$value = indefinite_article($type).' '.$type.' entity';
		}
		elseif(is_object($value)){
			$class = get_class($value);
			$value = 'an instance of the '.$class.' class';
		}
		elseif(is_resource($value))
			$value = 'a resource';
		elseif(!(is_int($value) || is_double($value) || is_long($value) || is_float($value)))
			$value = '"'.$value.'"';
			
		return $value;	
	}
	
	/**
	* Deprecated, use describe() instead.
	* @param mixed
	* @return string
	*/
	public function describe_as_text($value){
		return $this->describe($value);
	}
	
	/**
	* Describes multiple values in a way that would be useful in an error message.
	* Assumes that you're handing it an array of variable names mapping to variable values.
	* Lists the value for each variable using {@link describe}.
	* @param array Map of variable names to values.
	* @param string Glue to use when imploding the values to list in the error message.
	* @return string Description
	*/
	public function describe_values($values, $glue="\n"){
		$values = array_map(array($this, 'describe'), $values);
		$value_descriptions = array();
		foreach($values as $var_name => $value){
			$value_descriptions[] = $var_name.': '.$value;
		}
		return implode($glue, $value_descriptions);
	}	
	
	/**
	* Deprecated, use describe_values() instead.
	* @param array
	* @return string
	*/
	public function describe_values_as_text($values, $glue="\n"){ 
		return $this->describe_values($values, $glue);
	}
	
	/**
	* Internal method to determine whether or not a method name should map to {@link should_be_an_x}.
	* @param string
	* @return boolean
	*/
	protected function is_method_name_for_should_be_an_x($method_name){
		return mb_strpos($method_name, 'should_be_') === 0;
	}
	
	/**
	* Internal method to determine whether or not a method name should map to {@link property_should_be_an_x}.
	* @param string
	* @return boolean
	*/
	protected function is_method_name_for_property_value_should_be_an_x($method_name){
		return mb_strpos($method_name, 'property_value_should_be_') === 0;
	}
	
	/**
	* Triggers an error with the given message.
	*	
	* Note that this method sets a numeric value for a global called 
	*
	* @uses trigger_error
	* @uses level
	*
	* @param string The error message that should be displayed to the user.
	* @param string The named error level (should be one of the keys in {@link error_levels}
	* @param int The offset for the backtrace.  This expresses how far back in the backtrace the error handler should look to find the line where the user called on this class.
	* @return false
	*/
	protected function _trigger_error($message, $level, $offset=0){		

	 /** 
	 * Number of additional lines added to the backtrace by using the Error_helper to generate this error message.
	 * You can use this in your error handler to determine the file and line number where you triggered an error using the {@link Error_helper}.
	 * If you're not using an error handler that pays attention to this, errors generated using the {@link Error_helper} will always report themselves as occuring
	 * within the {@link _trigger_error} method at the point that {@link trigger_error} is called.
	 *
	 * If you're inside a Codeigniter context and using the {@link Icarus_Exceptions} class, this is already taken care of for you in the 
	 * {@link ICARUS_Exceptions::show_php_error} and {@link ICARUS_EXCEPTIONS::log_exception} methods.  If you're writing your own error handler, these methods will be a useful
	 * example for you.
	 *
	 * @global int $EXCEPTION_OFFSET
	 */
	 #BETTER WAY?  I HATE GLOBALS!
		global $EXCEPTION_OFFSET;
		$EXCEPTION_OFFSET = 1 + $offset;		
		
		trigger_error($message, $this->level($level));
		return false;
	}	
	
	function __call($method_name, $arguments){
		
		if($this->is_method_name_for_should_be_an_x($method_name)){
			$should_be = str_replace('should_be_an', '', $method_name);
			$should_be = str_replace('should_be_a', '', $should_be);
			$should_be = str_replace('should_be_', '', $should_be);

			//figure out if we were passed an offset
			$offset = 3;
			if(count($arguments) > 1 && is_numeric($arguments[1])){
				$offset += $arguments[1];
				unset($arguments[1]);
			}
						
			//turn the part after "should_be" into a message
			$arguments[] = implode(' ', explode('_', $should_be));
			//add the offset back to the end of the arguments
			$arguments[] = $offset;
			
			return call_user_func_array(array($this, 'should_be_an_x'), $arguments);
		}
		
		if($this->is_method_name_for_property_value_should_be_an_x($method_name)){
			$should_be = str_replace('property_value_should_be_an', '', $method_name);
			$should_be = str_replace('property_value_should_be_a', '', $should_be);
			$should_be = str_replace('property_value_should_be_', '', $should_be);

			//figure out if we were passed an offset
			$offset = 3;
			if(count($arguments) > 3 && is_numeric($arguments[3])){
				$offset += $arguments[3];
				unset($arguments[3]);
			}
						
			//turn the part after "should_be" into a message
			$arguments[] = implode(' ', explode('_', $should_be));
			//add the offset back to the end of the arguments
			$arguments[] = $offset;
			
			return call_user_func_array(array($this, 'property_value_should_be_an_x'), $arguments);
		}
		
		trigger_error('There is no method called '.$method_name.' for class Error_helper', E_USER_WARNING);
	}
}

